<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Repositories\Eloquent\Admin\RestaurantRepository;
use App\Http\Requests\Admin\RestaurantRequests\RestaurantStoreRequest;
use App\Http\Requests\Admin\RestaurantRequests\RestaurantUpdateRequest;
use App\Http\Requests\Api\RestaurantRequesrs\StoreRestaurantOrderRequest;
use App\Http\ServicesLayer\FairbaseServices\FairbaseService;
use App\Models\FavorateService;
use App\Models\Notification;
use App\Models\Restaurant;
use App\Models\RestaurantOrder;
use App\Models\WalletTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class RestaurantController extends Controller
{

    public $restaurant;
    public $restaurantRepository;
    public $restaurantOrder;
    public $favorateService;
    public $fairbaseService;
    public $notification;
    public $walletTransaction;

    public function __construct(
        Restaurant $restaurant, RestaurantRepository $restaurantRepository, RestaurantOrder $restaurantOrder, FavorateService $favorateService, 
        FairbaseService $fairbaseService, Notification $notification, WalletTransaction $walletTransaction
    )
    {
        $this->restaurant = $restaurant;
        $this->restaurantRepository = $restaurantRepository;
        $this->restaurantOrder = $restaurantOrder;
        $this->favorateService = $favorateService;
        $this->fairbaseService = $fairbaseService;
        $this->notification = $notification;
        $this->walletTransaction = $walletTransaction;
        $this->middleware('auth:api', ['except' => ['restaurants', 'details']]);
    }

    public function restaurants(Request $request, $offset, $limit)
    {
        try{
            $records = $this->restaurant
            ->when($request->country_id, function($q) use ($request) {
                $q->where('country_id', $request->country_id);
            })
            ->when($request->price_from && $request->price_to, function($q) use ($request) {
                $q->whereBetween('avg_prices', [$request->price_from, $request->price_to]);
            })
            ->when($request->price_from && !$request->price_to, function($q) use ($request) {
                $q->where('avg_prices', '>=', $request->price_from);
            })
            ->when(!$request->price_from && $request->price_to, function($q) use ($request) {
                $q->where('avg_prices', '<=', $request->price_to);
            })->unArchive()->orderBy('id', 'DESC')->withAvg('ratings', 'rating')->offset($offset)->limit(PAGINATION_COUNT)->get();
            return responseJson(200, "success", $records);
        }catch(\Exception $e){
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function details($id)
    {
        $record = $this->restaurant->where('id', $id)->withAvg('ratings', 'rating')->with(['ratings', 'days', 'times'])->first();
        return responseJson(200, "success", $record);
    }

    public function myItems(Request $request, $offset)
    {
        try{
            $auth = auth()->user();
            $records = $this->restaurant->where('owner_id', $auth->id)
            ->when($request->country_id, function($q) use ($request) {
                $q->where('country_id', $request->country_id);
            })
            ->when($request->price_from && $request->price_to, function($q) use ($request) {
                $q->whereBetween('avg_prices', [$request->price_from, $request->price_to]);
            })
            ->when($request->price_from && !$request->price_to, function($q) use ($request) {
                $q->where('avg_prices', '>=', $request->price_from);
            })
            ->when(!$request->price_from && $request->price_to, function($q) use ($request) {
                $q->where('avg_prices', '<=', $request->price_to);
            })->orderBy('id', 'DESC')->offset($offset)->limit(PAGINATION_COUNT)->get();
            return responseJson(200, "success", $records);
        }catch(\Exception $e){
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function myOrders(Request $request, $offset)
    {
        try{
            $auth = auth()->user();
            $records = $this->restaurantOrder->where('provider_id', $auth->id)
            ->when($request->status, function($q) use ($request) {
                $q->where('status', $request->status);
            })->with(['restaurant', 'user'])->orderBy('id', 'DESC')->offset($offset)
            ->limit(PAGINATION_COUNT)->get();
            return responseJson(200, "success", $records);
        }catch(\Exception $e){
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function orderStatus(Request $request, $id)
    {
        
        $validator = Validator::make($request->all(), [
            'status' => 'required|in:2,3,4,5,6',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }

        $auth = auth()->user();
        $order = $this->restaurantOrder->where('id', $id)->with(['user', 'chat_group.members.user' => function($q){
            $q->with(['wallet', 'group_transactions']);
        }])->first();
        // $order = $this->restaurantOrder->where('id', $id)->where('provider_id', $auth->id)->with('user')->first();
        if ($order) {

            if ((int)$request->status == 6 && (int)$order->status !== 2) {
                if ((int)$order->status == 1) {
                    return responseJson(404, "the provider has not been confirmed yet.");
                }
                return responseJson(404, "this action is not currently suitable for this request.");
            }
            if ((int)$request->status == 6) {
                if (!is_null($order->chat_group)) {
                    if ($order->chat_group?->added_by_id == $auth->id) {
                        
                        $errors = null;
                        $usersHaveErr = 0;

                        $n = $order->chat_group->members->count();
                        $costCents = (int) round($order->cost * 100);
                        $base = intdiv($costCents, $n);
                        $remainder = $costCents % $n;
                        
                        foreach ($order->chat_group->members as $i => $member) {
                            
                            $deposit = array_sum(array_column(
                                $member->user->group_transactions->where('chat_group_id', $order->chat_group_id)->where('type', 1)->where('status', 1)->toArray(), 
                                'amount'
                            ));
                            $withdrow = array_sum(array_column(
                                $member->user->group_transactions->where('chat_group_id', $order->chat_group_id)->where('type', 2)->where('status', 1)->toArray(), 
                                'amount'
                            ));
                            $balance = $deposit - $withdrow;
                            $shareCents = $base + ($i < $remainder ? 1 : 0);
                            $costPerPerson = $shareCents / 100;
                            if ($costPerPerson > $balance) {
                                $usersHaveErr++;
                                $errors .= $member->user->name . " - "; 
                            }

                            $transactions[] = [
                                'type' => 2, // withdrow
                                'status' => 1, // compeleted
                                'created_at' => now(), 'updated_at' => now(),
                                'chat_group_id' => $member->user->group_transactions[0]->chat_group_id ?? null,
                                'amount' => $costPerPerson, 'wallet_id' => $member->user->wallet?->id, 'user_id' => $member->user->id,
                                'orderable_id' => $id, 'orderable_type' => get_class($order),
                            ];
                        }
                        // if (!is_null($errors)) {
                        //     if($usersHaveErr > 1){ $errorMsg = 'users: '. $errors .'they not have balance'; }
                        //     else { $errorMsg = 'user: '. $errors .'not have balance'; }
                        //     return responseJson(404, $errorMsg);
                        // }
                    } else {
                        return responseJson(404, "you dont have permission, you must be the group owner");
                    }
                } else if ($order->added_by == $auth->id){
                    $auth->load(['personal_transactions']);
                    $deposit = array_sum(array_column(
                        $auth->user->personal_transactions->where('type', 1)->where('status', 1)->toArray(), 
                        'amount'
                    ));
                    $withdrow = array_sum(array_column(
                        $auth->user->personal_transactions->where('type', 2)->where('status', 1)->toArray(), 
                        'amount'
                    ));
                    $balance = $deposit - $withdrow;
                    if ($order->cost > $balance) {
                        return responseJson(404, "not have balance");
                    }
                    $transactions[] = [
                        'type' => 2, // withdrow
                        'status' => 1, // compeleted
                        'created_at' => now(), 'updated_at' => now(),
                        'amount' => $order->cost, 'wallet_id' => $auth->user->wallet->id, 'user_id' => $auth->user->id,
                        'orderable_id' => $id, 'orderable_type' => get_class($order),
                    ];
                } else {
                    return responseJson(404, "you dont have permission, you must be the group owner");
                }
            } else {
                if ($order->provider_id !== $auth->id) {
                    return responseJson(404, "not found");
                }
            }
            if ((int)$order->status == 3 || (int)$order->status == 4 || (int)$order->status == 5) {
                return responseJson(404, "this order is ended");
            }
            if ((int)$request->status == 3 && (int)$order->status == 1) {
                return responseJson(404, "must be confirmed first");
            }
            try{
                DB::beginTransaction();
                $order->update([
                    'status' => $request->status,
                    'canceled_by' => $request->status == 4 ? 1 : null,
                ]);
                if (count($transactions) > 0) {
                    $this->walletTransaction->insert($transactions);
                }
                DB::commit();
                
                $status = '';
                $request->status == 2 ? $status = "confirmed" : '';
                $request->status == 3 ? $status = "completed" : '';
                $request->status == 4 ? $status = "canceled" : '';
                if ($order->user?->fcm_token) {
                    // 1 => guides, 2 => cars, 3 => hotels, 4 => restaurants, 5 => cafes
                    $pushNotification = $this->fairbaseService->pushNotification(
                        $order->user?->fcm_token, 'change order status', "order number #$order->id has been $status", 4, $order->id
                    );
                }
                $this->notification->create([
                    'title' => 'change order status',
                    'content' => "order number #$order->id has been $status",
                    'order_id' => $order->id,
                    'chat_group_id' => $order->chat_group_id,
                    'provider_id' => $auth->id,
                    'user_id' => $order->added_by,
                    'serviceable_id' => $order->restaurant_id,
                    'serviceable_type' => "App\Models\Restaurant",
                ]);
                return responseJson(200, "success");
            }catch(\Exception $e){
                DB::rollBack();
                return responseJson(404, "There is something wrong , please contact technical support");
            }
        }else{
            return responseJson(404, "Not Found");
        }
    }

    public function makeOrder(StoreRestaurantOrderRequest $request)
    {
        try {
            $restaurant = $this->restaurant->where('id', $request->restaurant_id)->first();
            if ($restaurant) {
                $exists = $restaurant->times()->where('time', $request->table_time)->exists();
                if ($exists) {
                    $order = new $this->restaurantOrder();
                    
                    $order->provider_id = $restaurant->owner_id;
                    $order->restaurant_id = $request->restaurant_id;
                    $order->chat_group_id = $request->chat_group_id ?? null;
                    $order->trip_id = $request->trip_id ?? null;
                    $order->day = $request->day;
                    $order->table_time = $request->table_time;
                    $order->persons_count = $request->persons_count;
    
                    $order->added_by = auth()->user()->id;
                    $order->status = 1; // pending
                    $order->cost = $restaurant->avg_prices;
                    $order->save();

                    $this->notification->create([
                        'title' => 'new order',
                        'content' => "order number #$order->id",
                        'order_id' => $order->id,
                        'chat_group_id' => $order->chat_group_id,
                        'provider_id' => $restaurant->owner_id,
                        'user_id' => $order->added_by,
                        'serviceable_id' => $order->restaurant_id,
                        'serviceable_type' => "App\Models\Restaurant",
                    ]);
                    return responseJson(200, "success");
                }                
            }
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        } catch(\Exception $e){
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function getOrder($id)
    {
        $user = auth()->user();
        $record = $this->restaurantOrder->where('id', $id)->whereAny(['added_by', 'provider_id'], $user->id)->with([
            'user:id,name,img', 'restaurant', 'provider:id,name,email,img'
        ])->first();
        return responseJson(200, "success", $record);
    }

    public function addToFavorate(Request $request, $id)
    {
        $auth = auth()->user();
        $restaurant = $this->restaurant->where('id', $id)->first();
        if ($restaurant) {
            $this->favorateService->updateOrInsert(
                ['user_id' => $auth->id, 'serviceable_id' => $id, 'serviceable_type' => get_class($restaurant)],
                ['updated_at' => now()]
            );
            return responseJson(200, "success");
        }else{
            return responseJson(404, "Not Found");
        }
    }

    public function store(RestaurantStoreRequest $request)
    {
        try{
            $request->merge(['owner_id' => auth()->user()->id]);
            $this->restaurantRepository->store($request);
            return responseJson(200, "success");
        }catch(\Exception $e){
            return responseJson(404, "There is something wrong , please contact technical support");
        }
    }

    public function update(RestaurantUpdateRequest $request, $id)
    {
        $auth = auth()->user();
        $restaurant = $this->restaurant->where('id', $id)->where('owner_id', $auth->id)->first();
        if ($restaurant) {
            try{
                $this->restaurantRepository->update($request, $id);
                return responseJson(200, "success");
            }catch(\Exception $e){
                DB::rollBack();
                return responseJson(404, "There is something wrong , please contact technical support");
            }
        }else{
            return responseJson(404, "Not Found");
        }
    }

    public function activate($id)
    {
        $auth = auth()->user();
        $restaurant = $this->restaurant->where('id', $id)->where('owner_id', $auth->id)->first();
        if ($restaurant) {
            try{
                $this->restaurantRepository->activate($id);
                return responseJson(200, "success");
            }catch(\Exception $e){
                return responseJson(404, "There is something wrong , please contact technical support");
            }
        }else{
            return responseJson(404, "Not Found");
        }
    }

}
