<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Repositories\Eloquent\Admin\DestinationRepository;
use App\Http\Repositories\Eloquent\Admin\WalletRepository;
use App\Http\Requests\Admin\DestinationRequests\DestinationStoreRequest;
use App\Http\Requests\Admin\DestinationRequests\DestinationUpdateRequest;
use App\Models\Destination;
use App\Models\FavorateService;
use App\Models\WalletTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DestinationController extends Controller
{

    public $destination;
    public $destinationRepository;
    public $favorateService;
    public $walletTransaction;
    public $walletRepository;

    public function __construct(
        Destination $destination, DestinationRepository $destinationRepository, FavorateService $favorateService, 
        WalletTransaction $walletTransaction, WalletRepository $walletRepository
    )
    {
        $this->destination = $destination;
        $this->destinationRepository = $destinationRepository;
        $this->favorateService = $favorateService;
        $this->walletTransaction = $walletTransaction;
        $this->walletRepository = $walletRepository;
        $this->middleware('auth:api', ['except' => ['destinations', 'details']]);
    }

    public function destinations(Request $request, $offset, $limit)
    {
        try{
            $records = $this->destination
            ->when($request->country_id, function($q) use ($request) {
                $q->where('country_id', $request->country_id);
            })
            ->when($request->price_from && $request->price_to, function($q) use ($request) {
                $q->whereBetween('day_price', [$request->price_from, $request->price_to]);
            })
            ->when($request->price_from && !$request->price_to, function($q) use ($request) {
                $q->where('day_price', '>=', $request->price_from);
            })
            ->when(!$request->price_from && $request->price_to, function($q) use ($request) {
                $q->where('day_price', '<=', $request->price_to);
            })->unArchive()->orderBy('id', 'DESC')->withAvg('ratings', 'rating')->offset($offset)->limit(PAGINATION_COUNT)->get();
            return responseJson(200, "success", $records);
        }catch(\Exception $e){
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function details($id)
    {
        $record = $this->destination->where('id', $id)->withAvg('ratings', 'rating')->with(['ratings'])->first();
        return responseJson(200, "success", $record);
    }

    public function myItems(Request $request, $offset)
    {
        try{
            $auth = auth()->user();
            $records = $this->destination->where('owner_id', $auth->id)
            ->when($request->country_id, function($q) use ($request) {
                $q->where('country_id', $request->country_id);
            })
            ->when($request->price_from && $request->price_to, function($q) use ($request) {
                $q->whereBetween('day_price', [$request->price_from, $request->price_to]);
            })
            ->when($request->price_from && !$request->price_to, function($q) use ($request) {
                $q->where('day_price', '>=', $request->price_from);
            })
            ->when(!$request->price_from && $request->price_to, function($q) use ($request) {
                $q->where('day_price', '<=', $request->price_to);
            })->orderBy('id', 'DESC')->offset($offset)->limit(PAGINATION_COUNT)->get();
            return responseJson(200, "success", $records);
        }catch(\Exception $e){
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function addToFavorate(Request $request, $id)
    {
        $auth = auth()->user();
        $destination = $this->destination->where('id', $id)->first();
        if ($destination) {
            $this->favorateService->updateOrInsert(
                ['user_id' => $auth->id, 'serviceable_id' => $id, 'serviceable_type' => get_class($destination)],
                ['updated_at' => now()]
            );
            return responseJson(200, "success");
        }else{
            return responseJson(404, "Not Found");
        }
    }

    public function store(DestinationStoreRequest $request)
    {
        try{
            $request->merge(['owner_id' => auth()->user()->id]);
            $this->destinationRepository->store($request);
            return responseJson(200, "success");
        }catch(\Exception $e){
            return responseJson(404, "There is something wrong , please contact technical support");
        }
    }

    public function update(DestinationUpdateRequest $request, $id)
    {
        $auth = auth()->user();
        $destination = $this->destination->where('id', $id)->where('owner_id', $auth->id)->first();
        if ($destination) {
            try{
                $this->destinationRepository->update($request, $id);
                return responseJson(200, "success");
            }catch(\Exception $e){
                DB::rollBack();
                return responseJson(404, "There is something wrong , please contact technical support");
            }
        }else{
            return responseJson(404, "Not Found");
        }
    }

    public function activate($id)
    {
        $auth = auth()->user();
        $destination = $this->destination->where('id', $id)->where('owner_id', $auth->id)->first();
        if ($destination) {
            try{
                $this->destinationRepository->activate($id);
                return responseJson(200, "success");
            }catch(\Exception $e){
                return responseJson(404, "There is something wrong , please contact technical support");
            }
        }else{
            return responseJson(404, "Not Found");
        }
    }

}
