<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\ServicesLayer\FairbaseServices\FairbaseService;
use App\Models\ChatGroup;
use App\Models\ChatGroupMedia;
use App\Models\ChatGroupMember;
use App\Models\ChatGroupMessage;
use App\Models\ChatGroupMessageOption;
use App\Models\ChatGroupMessageOptionVote;
use App\Models\User;
use App\Models\Hotel;
use App\Models\Cafe;
use App\Models\Restaurant;
use App\Models\Destination;
use App\Models\Car;
use App\Models\Interaction;
use App\Models\Trip;
use App\Models\Wallet;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;

class ChatController extends Controller
{
 
    public $user;
    public $chatGroup;
    public $chatGroupMember;
    public $chatGroupMessage;
    public $fairbaseService;
    public $chatGroupMessageOption;
    public $chatGroupMessageOptionVote;
    public $chatGroupMedia;
    public $wallet;
    public $trip;
    public $interaction;

    public function __construct(
        User $user, ChatGroup $chatGroup, ChatGroupMember $chatGroupMember, ChatGroupMessage $chatGroupMessage, FairbaseService $fairbaseService,
        ChatGroupMessageOption $chatGroupMessageOption, ChatGroupMessageOptionVote $chatGroupMessageOptionVote, ChatGroupMedia $chatGroupMedia,
        Wallet $wallet, Trip $trip, Interaction $interaction
    ){
        $this->user = $user;
        $this->chatGroup = $chatGroup;
        $this->chatGroupMember = $chatGroupMember;
        $this->chatGroupMessage = $chatGroupMessage;
        $this->fairbaseService = $fairbaseService;
        $this->chatGroupMessageOption = $chatGroupMessageOption;
        $this->chatGroupMessageOptionVote = $chatGroupMessageOptionVote;
        $this->chatGroupMedia = $chatGroupMedia;
        $this->wallet = $wallet;
        $this->trip = $trip;
        $this->interaction = $interaction;
        $this->middleware('auth:api', ['except' => []]);
    }

    public function messages(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'group_id' => 'required|exists:chat_groups,id',
            'message' => 'required_without:file|string',
            'file'   => 'required_without:message|file|mimes:jpg,jpeg,png,mp4,mp3,wav|max:20480',
            'message_type'   => 'required_with:file|max:255',
            'message_options' => 'nullable|array',
            'message_options.*' => 'required|string',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors());  
        }

        try {
            DB::beginTransaction();
            $file = null;
            if (!$request->hasFile('file') == null) {
                $file = uploadIamge($request->file('file'), 'chat'); // function on helper file to upload file
                $mediaArr [] = [
                    'file' => $file,
                    'created_at' => now(),
                    'updated_at' => now(),
                    'group_id' => $request->group_id,
                    'added_by_id' => auth()->user()->id,
                ];
                $this->chatGroupMedia->insert($mediaArr);
            }
            $data = [];
            $group = $this->chatGroup->where('id', $request->group_id)->active()->first();
            $messageData = $this->chatGroupMessage->create([
                'created_at' => now(),
                'updated_at' => now(),
                'file' => $file,
                'message_type' => $request->message_type,
                'message' => $request->message ?? null,
                'group_id' => $request->group_id,
                'added_by_id' => auth()->user()->id,
                'is_multiple' => $request->is_multiple
            ]);
            if (isset($request->message_options) && count($request->message_options) > 0) {
                $options = [];
                foreach ($request->message_options as $option) {
                    $options [] = [
                        'created_at' => now(),
                        'updated_at' => now(),
                        'option' => $option,
                        'message_id' => $messageData->id,
                    ];
                }
                $this->chatGroupMessageOption->insert($options);
            }
            $this->fairbaseService->pushNotification("chat", "new message in group: $group->name", $request->message ?? '', 1, $request->group_id);
            $this->fairbaseService->saveChatMessage($messageData, 1, $group->id, null);
            DB::commit();
            
            $data['message'] = $messageData;
            $message = ltrim($request->message);
            if (str_starts_with($message, '/') || str_starts_with($message, '\\')) {
                $data['ai_response'] = $this->sendToAI($request);
            }
            return responseJson(200, "success", $data);
        } catch (\Exception $e) {
            DB::rollBack();
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
        
    }

    public function readMessages(Request $request, $id)
    {
        $group = $this->chatGroup->whereHas('members', function ($q) {
            $q->where('user_id', auth()->id());
        })->where('id', $id)->first();
        try {
            $this->markAllUpToLatest($group->id);
        } catch (\Exception $e) {}
        return responseJson(200, "success");
    }

    public function messageVotes(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'option_ids' => 'required|array',
            'option_ids.*' => 'required|exists:chat_group_message_options,id',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors());
        }
        
        $data = [];
        foreach ($request->option_ids as $option_id) {
            $data[] = [
                'created_at' => now(),
                'updated_at' => now(),
                'option_id' => $option_id,
                'user_id' => auth()->user()->id,
            ];
        }
        $this->chatGroupMessageOptionVote->upsert($data, ['option_id', 'user_id'], ['updated_at']);
        return responseJson(200, "success");
    }

    public function checkUsers(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'users_arr' => 'required|array',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors());
        }
        $usersArr = $request->users_arr;
        $query = DB::table('users')->select('id', 'mobile', 'fcm_token')->whereIn('mobile', $usersArr)->get()->toArray();
        return responseJson(200, "success", $query);
    }

    public function groups()
    {
        $groups = $this->chatGroup->whereHas('members', function ($q) {
            $q->where('user_id', auth()->id());
        })->active()->unArchive()->with('last_message.user')
        ->withCount(['unreadMessages as unread_for_me'])->get();
        return responseJson(200, "success", $groups);
    }

    public function group($id, $messages_page = 0)
    {
        $relations = [];
        if (request()->filled('relation')) {
            
            switch (request()->get('relation')) {
                case 'members':
                    $relations = ['members:id,group_id,added_by_id,user_id', 'members.user:id,name,img,user_type'];
                    break;

                case 'media':
                    $relations = ['media.commentable'];
                    break;

                case 'messages':
                    $relations = ['messages' => function ($query) use($messages_page) {
                        $query->with(['user:id,name,img,user_type', 'options' => function ($subQuery) {
                            $subQuery->withCount('votes');
                        }])->latest()->offset(PAGINATION_COUNT * $messages_page)->limit(PAGINATION_COUNT);
                    }];
                    break;

                case 'orders':
                    $relations = [
                        'car_orders' => function ($query) {
                            $query->with(['user:id,name,img,user_type', 'car', 'provider:id,name,email,img']);
                        }, 
                        'cafe_orders' => function ($query) {
                            $query->with(['user:id,name,img,user_type', 'cafe', 'provider:id,name,email,img']);
                        }, 
                        'hotel_orders' => function ($query) {
                            $query->with(['user:id,name,img,user_type', 'hotel', 'provider:id,name,email,img']);
                        }, 
                        'guide_orders' => function ($query) {
                            $query->with(['user:id,name,img,user_type', 'guide']);
                        }, 
                        'restaurant_orders' => function ($query) {
                            $query->with(['user:id,name,img,user_type', 'restaurant', 'provider:id,name,email,img']);
                        }
                    ];
                    break;

                case 'wallet':
                    $relations = ['wallet.transactions'];
                    break;
                    
                case 'trips':
                    $relations = ['trips'];
                    break;
                    
                default:
                    break;
            }
        } else {
            $relations = [
                'members:id,group_id,added_by_id,user_id', 'members.user:id,name,img,user_type',
                'media', 'wallet.transactions', 'trips',
                'messages' => function ($query) use($messages_page) {
                    $query->with(['user:id,name,img,user_type', 'options' => function ($subQuery) {
                        $subQuery->withCount('votes');
                    }])->latest()->offset(PAGINATION_COUNT * $messages_page)->limit(PAGINATION_COUNT);
                },
                'car_orders' => function ($query) {
                    $query->with(['user:id,name,img,user_type', 'car', 'provider:id,name,email,img']);
                }, 
                'cafe_orders' => function ($query) {
                    $query->with(['user:id,name,img,user_type', 'cafe', 'provider:id,name,email,img']);
                }, 
                'hotel_orders' => function ($query) {
                    $query->with(['user:id,name,img,user_type', 'hotel', 'provider:id,name,email,img']);
                }, 
                'guide_orders' => function ($query) {
                    $query->with(['user:id,name,img,user_type', 'guide']);
                }, 
                'restaurant_orders' => function ($query) {
                    $query->with(['user:id,name,img,user_type', 'restaurant', 'provider:id,name,email,img']);
                },
                // 'messages.user:id,name,img,user_type', 'messages.options'
            ];
        }

        $group = $this->chatGroup->whereHas('members', function ($q) {
            $q->where('user_id', auth()->id());
        })->where('id', $id)->active()->with($relations)->first();
        try {
            $this->markAllUpToLatest($group->id);
        } catch (\Exception $e) {}
        
        return responseJson(200, "success", $group);
    }

    public function trip($id)
    {
        $trip = $this->trip->where('id', $id)->active()->with([
            'car_orders' => function ($query) {
                $query->with(['user:id,name,img,user_type', 'car', 'provider:id,name,email,img']);
            }, 
            'cafe_orders' => function ($query) {
                $query->with(['user:id,name,img,user_type', 'cafe', 'provider:id,name,email,img']);
            }, 
            'hotel_orders' => function ($query) {
                $query->with(['user:id,name,img,user_type', 'hotel', 'provider:id,name,email,img']);
            }, 
            'restaurant_orders' => function ($query) {
                $query->with(['user:id,name,img,user_type', 'restaurant', 'provider:id,name,email,img']);
            },
        ])->first();
        return responseJson(200, "success", $trip);
    }

    public function addGroup(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'nullable|image',
            'name' => 'required|string|max:255',
            'destination' => 'required|string|max:1255',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        try {

            $file = null;
            $user = auth()->user();
            if (!$request->hasFile('file') == null) {
                $file = uploadIamge($request->file('file'), 'chat'); // function on helper file to upload file
            }
            $group = $this->chatGroup->create([
                'created_at' => now(),
                'updated_at' => now(),
                'img' => $file,
                'name' => $request->name,
                'destination' => $request->destination,
                'added_by_id' => $user->id,
            ]);
            $this->chatGroupMember->create([
                'created_at' => now(),
                'updated_at' => now(),
                'user_id' => $user->id,
                'group_id' => $group->id, 
                'added_by_id' => $user->id,
            ]);
            return responseJson(200, "success", $group);
        } catch (\Exception $e) {
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function addTrip(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'chat_group_id' => 'required|exists:chat_groups,id',
            'duration' => 'required_without_all:from,to|max:255',
            'from' => 'required_without:duration|date',
            'to' => 'required_without:duration|date',
            'file' => 'nullable|image',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        try {

            $file = null;
            $user = auth()->user();
            if (!$request->hasFile('file') == null) {
                $file = uploadIamge($request->file('file'), 'chat'); // function on helper file to upload file
            }
            $duration = $request->duration;
            $to = isset($request->to) && !is_null($request->to) ? Carbon::parse($request->to) : null;
            $from = isset($request->from) && !is_null($request->from) ? Carbon::parse($request->from) : null;
            if (is_null($request->duration)) {
                $duration = $from->diffInDays($to);
            }
            $this->trip->create([
                'created_at' => now(),
                'updated_at' => now(),
                'img' => $file,
                'name' => $request->name,
                'chat_group_id' => $request->chat_group_id,
                'duration' => $duration ?? null,
                'from' => $from ?? null,
                'to' => $to ?? null,
                'added_by' => $user->id,
            ]);
            return responseJson(200, "success");
        } catch (\Exception $e) {
            // dd($e);
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function removeGroup(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'group_id' => 'required|exists:chat_groups,id',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        try {
            DB::beginTransaction();

            $group = $this->chatGroup->where('id', $request->group_id)->with(['members.user'])->first();
            $tokens = $group->members->pluck('user')->filter(fn($user) => $user && $user->fcm_token)->map(fn($user) => [
                'id' => $user->id,
                'fcm_token' => $user->fcm_token,
            ])->values()->toArray();
            $group->is_activate = 0;
            $group->deleted_at = now();
            $group->save();
            
            DB::commit();
            return responseJson(200, "success", $tokens);
        } catch (\Exception $e) {
            DB::rollBack();
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function addMembers(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'group_id' => 'required|exists:chat_groups,id',
            'user_ids' => 'required|array',
            'user_ids.*' => 'required|exists:users,id',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        try {

            $groupId = $request->group_id;
            $userIds = $request->input('user_ids');
        
            $data = [];
            foreach ($userIds as $userId) {
                $data[] = [
                    'group_id' => $groupId,
                    'user_id' => $userId,
                    'updated_at' => now(),
                    'added_by_id' => auth()->user()->id,
                ];
            }
            $this->chatGroupMember->upsert($data, ['group_id', 'user_id'], ['updated_at']);
            return responseJson(200, "success");
        } catch (\Exception $e) {
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function membersWallet($id)
    {
        try {
            $users = $this->getMembersWallet($id);
        } catch (\Exception $e) {
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
        return responseJson(200, "success", $users);
    }

    public function getMembersWallet($id)
    {
        try {
            $group = ChatGroup::where('id', $id)->with(['members.user' => function($q) use($id) {
                $q->with(['transactions' => fn($qq) => $qq->where('chat_group_id', $id)->where('status', 1)]);
            }])->first();
            $users = $group->members->pluck('user');
            foreach ($users as $user) {
                $user->depositsData = $user->transactions->where('type', 1)->sum('amount');
                $user->withdrawalsData = $user->transactions->where('type', 2)->sum('amount');
            }
            return $users;
        } catch (\Exception $e) {
            return [];
        }
    }

    public function removeMembers(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'group_id' => 'required|exists:chat_groups,id',
            'user_id' => 'required|exists:users,id',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        try {
            $user = $this->user->select('id', 'mobile', 'fcm_token')->where('id', $request->user_id)->first();
            $this->chatGroupMember->where('group_id', $request->group_id)->where('user_id', $request->user_id)->delete();
            return responseJson(200, "success", $user);
        } catch (\Exception $e) {
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function addMedia(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'files' => 'required|array',
            'files.*' => 'required|file|mimes:jpeg,jpg,png,webp,mp4,avi,mov,mpeg|max:51200',
            'group_id' => 'required|exists:chat_groups,id',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        try {

            foreach ($request->file('files') as $file) {
                $fileName = uploadIamge($file, 'chat'); // function on helper file to upload file
                $mediaArr [] = [
                    'created_at' => now(),
                    'updated_at' => now(),
                    'file' => $fileName,
                    'group_id' => $request->group_id,
                    'added_by_id' => auth()->user()->id,
                ];
            }
            $this->chatGroupMedia->insert($mediaArr);
            return responseJson(200, "success");
        } catch (\Exception $e) {
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function removeMedia(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file_id' => 'required|exists:chat_group_media,id',
            'group_id' => 'required|exists:chat_groups,id',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        try {
            $this->chatGroupMedia->where('group_id', $request->group_id)->where('id', $request->file_id)->delete();
            return responseJson(200, "success");
        } catch (\Exception $e) {
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function addMediaInteraction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'react' => 'nullable|integer|max:8|min:1',
            'comment' => 'nullable|string|max:1255',
            'file_id' => 'required|exists:chat_group_media,id',
            'group_id' => 'required|exists:chat_groups,id',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        try {
            $user = auth()->user();
            $media = $this->chatGroupMedia->where('group_id', $request->group_id)->where('id', $request->file_id)->first();
            if (isset($request->react) || isset($request->comment) && !is_null($media)) {
                $this->interaction->create([
                    'commentable_id' => $media->id,
                    'commentable_type' => get_class($media),
                    'user_id' => $user->id,
                    'react' => $request->react ?? null,
                    'comment' => $request->comment ?? null,
                ]);
            }
            return responseJson(200, "success");
        } catch (\Exception $e) {
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
    }

    public function sendToAI(Request $request)
    {

        $schema = <<<EOT
            Tables:
            - hotels(id, name, info, location, country_id, average_price) => Model: \App\Models\Hotel
            - restaurants(id, name, info, location, country_id, average_price) => Model: \App\Models\Restaurant
            - cafes(id, name, info, location, country_id, average_price) => Model: \App\Models\Cafe
            - destinations(id, name, info, location, country_id, type) => Model: \App\Models\Destination
            - cars(id, name, country_id, average_price) => Model: \App\Models\Car
            - guides(id, name, country_id, experience_years) => Model: \App\Models\User
            - countries(id, name) => Model: \App\Models\Country

            Instructions:
            1. The user input might be a country, city, or region.
            2. Use only the columns listed above for each table. DO NOT use any fields or attributes not explicitly listed in the schema.
            3. If input is a **country**, use:
            `whereHas('country', fn(q) => q->where('name', 'LIKE', '%CountryName%'))`
            4. If input is a **city or region**, use your **geographic knowledge** to identify the **country it belongs to**.
            For example: "الرياض" => "السعودية", "ميلانو" => "إيطاليا".
            5. Then use the **country name** in `whereHas(...)`.  
            6. Always apply the `whereHas('country', ...)` condition **first**.
            7. Do **not** use `location` or any similar field that is not listed in the table schema.
            8. If the user mentions price filtering, apply it **once** and only if `average_price` exists in the table:
            - Use `whereBetween('average_price', [Min, Max])`, or  
            - `where('average_price', '<=', MaxValue)`, or  
            - `where('average_price', '>=', MaxValue)` depending on context.
            9. Use Laravel Eloquent syntax (PHP code), no raw SQL.
            10. Return **only** the Eloquent query, ending with:  
                `->unArchive()->orderBy('id', 'DESC')->offset(0)->limit(10)->get();`  
                — no explanations, no variable assignment, no comments.
            11. Your response must look **exactly** like the example provided.
        EOT;

        $userPrompt = $request->message;
        // $userPrompt = "عاوز فنادق في السعوديه";
        $finalPrompt = "$schema\n\nUser question: $userPrompt\nSQL:";

        try {
            $response = Http::withToken("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")
                ->post('https://api.openai.com/v1/chat/completions', [
                    'model' => 'gpt-4',
                    'messages' => [
                        ['role' => 'user', 'content' => $finalPrompt]
                    ],
                    'temperature' => 0,
                ]
            );
            $sql = $response['choices'][0]['message']['content'] ?? null;

            // dd($sql);
            $codeToRun = "return $sql";
            // dd($codeToRun);
            $data = eval($codeToRun);
            // dd($data);
            
            return $data;
            return responseJson(200, "success", $data);
        } catch (\Exception $e) {
            return null;
            return responseJson(500, "there is some thing wrong , please contact technical support . {{$e->getMessage()}}");
        }
    }

    public static function markAllUpToLatest(int $groupId) 
    {
        $now = now();
        $userId = auth()->id();

        $latestRow = DB::table('chat_group_messages')->where('group_id', $groupId)
        ->whereNull('deleted_at')->orderByDesc('id')->select('id')->first();

        $latestId = $latestRow->id ?? 0;
        if ($latestId === 0) {
            return 1;
        }

        $lastReadBefore = (int) optional(
            DB::table('chat_group_message_reads as r')
            ->join('chat_group_messages as m', 'm.id', '=', 'r.message_id')
            ->where('m.group_id', $groupId)->where('r.user_id', $userId)
            ->orderByDesc('r.seen_at')->orderByDesc('m.id')->select('m.id')->first()
        )->id ?? 0;

        if ($lastReadBefore >= $latestId) {
            DB::table('chat_group_members')->updateOrInsert(
                ['group_id' => $groupId, 'user_id' => $userId],
                ['last_read_message_id' => $latestId, 'updated_at' => $now]
            );
            return 1;
        }

        $markedCount = 0;
        $lastMarkedId = null;
        $firstMarkedId = null;

        DB::table('chat_group_messages')->where('group_id', $groupId)->where('id', '>', $lastReadBefore)
        ->where('id', '<=', $latestId)->whereNull('deleted_at')->orderBy('id')
        ->chunkById(1000, function ($rows) use ($userId, $now, &$markedCount, &$firstMarkedId, &$lastMarkedId) {
            $payload = [];
            foreach ($rows as $r) {
                $payload[] = [
                    'message_id' => $r->id,
                    'user_id'    => $userId,
                    'seen_at'    => $now,
                ];
                if ($firstMarkedId === null) $firstMarkedId = $r->id;
                $lastMarkedId = $r->id;
            }
            if (! empty($payload)) {
                DB::table('chat_group_message_reads')->upsert(
                    $payload,
                    ['message_id', 'user_id'],
                    ['seen_at']
                );
                $markedCount += count($payload);
            }
        });

        $lastReadAfter = $latestId;
        DB::table('chat_group_members')->updateOrInsert(
            ['group_id' => $groupId, 'user_id' => $userId],
            ['last_read_message_id' => $lastReadAfter, 'updated_at' => $now]
        );
        return 1;
    }

}
